// File: ex_insertion.cpp

#include <list>
#include <boost/lexical_cast.hpp>
#include <boost/timer.hpp>

#include "bops_set_linear.h"
#include "pgn_print.h"

// Generate a "pyramid" made of n(n+1)/2 (approximately) equilateral triangles.
// The base of each triangle is 2 and the height is sqrt(3) ~ 1.732.
template <typename Output_iterator>
Output_iterator generate(unsigned int n, Output_iterator oi)
{
  for (unsigned int j = 0; j != n; ++j) {
    for (unsigned int i = 0; i != n-j; ++i) {
      Polygon polygon;
      Kernel::FT yb(CGAL::Gmpq(1732 * j, 1000));
      Kernel::FT yt(CGAL::Gmpq(1732 * (j + 1), 1000));
      polygon.push_back(Point(Kernel::FT(j + 2 * i), yb));
      polygon.push_back(Point(Kernel::FT(j + 2 * i + 2), yb));
      polygon.push_back(Point(Kernel::FT(j + 2 * i + 1), yt));
      *oi++ = polygon;
    }
  }
  return oi;
}

int main(int argc, char* argv[])
{
  unsigned int n = (argc >= 2) ? boost::lexical_cast<unsigned int>(argv[1]) : 4;
  std::list<Polygon> polygons;
  generate(n, std::back_inserter(polygons));

  std::list<Polygon>::const_iterator it;
  Polygon_set R1, R2;

  boost::timer timer;
  for (it = polygons.begin(); it != polygons.end(); ++it)
    R1.insert(Polygon_with_holes(*it));
  double secs = timer.elapsed();
  std::cout << "Insertion took " << secs << " seconds." << std::endl;
  
  timer.restart();
  for (it = polygons.begin(); it != polygons.end(); ++it) R2.join(*it);
  secs = timer.elapsed();
  std::cout << "Union took " << secs << " seconds." << std::endl;

  print_polygon_set(R1);
  print_polygon_set(R2);
  return 0;
}
